/*--------------------------------------------------------------------------*\

    FILE....: FIFOC.C
    TYPE....: C Functions
    AUTHOR..: David Rowe
    DATE....: 6/10/97

    Functions used to implement First In First Out (FIFO) queues.  This 
    version is C-callable and is used with the OpenSwitch12 kernel
    mode driver hda.c

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         V12PCI CT Card Software

         Copyright (C) David Rowe 2001 david@voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

#include "fifoc.h"

/*--------------------------------------------------------------------------*\

				TYPEDEFS

\*--------------------------------------------------------------------------*/

typedef struct {
	char   *pstart;		// first char in FIFO			
	char   *pend;		// one after last char in FIFO		
	char   *pwr;		// write pointer			
	char   *prd; 		// read pointer				
	int     size;		// total storage in FIFO		
} FIFO;

/*--------------------------------------------------------------------------*\

				FUNCTIONS

\*--------------------------------------------------------------------------*/

/*--------------------------------------------------------------------------*\

	FUNCTION.: fifo_create
	AUTHOR...: David Rowe
	DATE.....: 6/10/97

	Creates a FIFO.

\*--------------------------------------------------------------------------*/

void fifo_create(void **ppvfifo, int  size)
{
    FIFO   *fifo;

    *ppvfifo = (void*)fifo_malloc(sizeof(FIFO));
    fifo = (FIFO*)(*ppvfifo);

    fifo->pstart = (char*)fifo_malloc(sizeof(char)*size);
    fifo->pend = fifo->pstart + size;

    fifo->pwr = fifo->pstart;
    fifo->prd = fifo->pstart;
    fifo->size = size;
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: fifo_destroy
	AUTHOR...: David Rowe
	DATE.....: 6/10/97

	Destroys a previously created FIFO.

\*--------------------------------------------------------------------------*/

void fifo_destroy(void *pvfifo)
{
    FIFO   *fifo;

    fifo = (FIFO*)pvfifo;
    fifo_free(fifo->pstart);
    fifo_free(fifo);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: fifo_write
	AUTHOR...: David Rowe
	DATE.....: 6/10/97

	Writes a block of chars to a FIFO.  Returns OK if successful.

\*--------------------------------------------------------------------------*/

int fifo_write(void *pvfifo, char *buf, int size)
{
    FIFO   *fifo;
    int    chars_used;	// used space in FIFO				 
    int    chars_free;	// free space in FIFO				 
    int    copy_first;	// size of first block move			 
    char   *new_pwr;	// modified pwr				 
    char   *tmp;
    char   *prd;

    fifo = (FIFO*)pvfifo;

    // determine if there is data in FIFO to fill buf

    prd = fifo->prd;
    if (fifo->pwr >= prd)
	chars_used = fifo->pwr - prd;
    else
	chars_used = fifo->size - (prd - fifo->pwr);
    chars_free = fifo->size - chars_used - 1;
    if (chars_free < size)
	return(FIFO_FULL);

    // If buf overlaps end of linear array split into two block moves

    if ((fifo->pwr + size) > fifo->pend) {
	copy_first = (fifo->pend-fifo->pwr);

	fifo_memcpy(fifo->pwr, buf, copy_first*sizeof(char));
        fifo_memcpy(fifo->pstart, &buf[copy_first], 
		    (size-copy_first)*sizeof(char));
    }
    else
	fifo_memcpy(fifo->pwr, buf, size*sizeof(char));

    // increment pwr and wrap around if required

    new_pwr = fifo->pwr + size;
    if (new_pwr >= fifo->pend) {
	tmp = fifo->pstart + (new_pwr - fifo->pend);
    }
    else {
	tmp = new_pwr;
    }

    fifo->pwr = tmp;
    return(OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: fifo_read
	AUTHOR...: David Rowe
	DATE.....: 6/10/97

	Reads a block of chars from a FIFO.  Returns OK if successful.

\*--------------------------------------------------------------------------*/

int fifo_read(void *pvfifo, char *buf, int size)
{
    FIFO   *fifo;
    int    copy_first;	// size of first copy				 
    int    chars_used;	// used space in FIFO				 
    char   *new_prd;	// modified prd					 

    fifo = (FIFO*)pvfifo;

    // determine if there is data in FIFO for buf

    if (fifo->pwr >= fifo->prd)
	chars_used = fifo->pwr - fifo->prd;
    else
	chars_used = fifo->size - (fifo->prd - fifo->pwr);
    if (chars_used < size)
	return(FIFO_EMPTY);

    // If buf overlaps end of linear array split into two block moves

    if ((fifo->prd + size) > fifo->pend) {
	copy_first = (fifo->pend-fifo->prd);

	fifo_memcpy(buf, fifo->prd, copy_first*sizeof(char));
	fifo_memcpy(&buf[copy_first], fifo->pstart, 
		    (size-copy_first)*sizeof(char));
    }
    else
	fifo_memcpy(buf, fifo->prd, size*sizeof(char));

    // increment prd and wrap around if required

    new_prd = fifo->prd + size;
    if (new_prd >= fifo->pend)
	fifo->prd = fifo->pstart + (new_prd - fifo->pend);
    else
	fifo->prd = new_prd;

    return(OK);
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: fifo_how_full
	AUTHOR...: David Rowe
	DATE.....: 2/10/97

	Determines number of used chars in FIFO.

\*--------------------------------------------------------------------------*/

void fifo_how_full(void *pvfifo, int *chars)
{
    FIFO   *fifo;

    fifo = (FIFO*)pvfifo;

    if (fifo->pwr >= fifo->prd)
	*chars = fifo->pwr - fifo->prd;
    if (fifo->prd > fifo->pwr)
	*chars = fifo->size - (fifo->prd - fifo->pwr);
}

